﻿using System;
using System.Threading;
using FyndSharp.Utilities.Collections;
using gov.va.med.VBECS.Communication.Channels;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Server
{
    internal abstract class BaseServer : IServer
    {
        /// <summary>
        /// Used to set an auto incremental unique identifier to clients.
        /// </summary>
        private static long _lastClientId;
        public event EventHandler<ThreadExceptionEventArgs> FatalErrorOccured;
        private IListener _listener;

        protected BaseServer()
        {
            ClientDummies = new SynchronizedSortedList<long, IClientDummy>();
            ProtocolFactory = ProtocolManager.Instance().ProtocolFactory;
        }

        #region IServer Members

        public event EventHandler<ClientDummyEventArgs> ClientConnected;
        public event EventHandler<ClientDummyEventArgs> ClientDisconnected;

        public SynchronizedSortedList<long, IClientDummy> ClientDummies { get; private set; }
        public IProtocolFactory ProtocolFactory { get; set; }

        public virtual void Start()
        {
            _listener = CreateListener();
            _listener.ChannelConnected += listener_channel_connected;
            _listener.Start();
        }

        public virtual void Stop()
        {
            if (null != _listener)
            {
                _listener.Stop();
            }
        }

        #endregion

        /// <summary>
        /// Gets an unique number to be used as identifier of a client.
        /// </summary>
        /// <returns>Client ID</returns>
        public static long GetClientId()
        {
            return Interlocked.Increment(ref _lastClientId);
        }

        protected abstract IListener CreateListener();

        private void listener_channel_connected(object sender, ChannelEventArgs e)
        {
            IClientDummy clientDummy = new ClientDummy(GetClientId(), e.Channel);
            clientDummy.Protocol = ProtocolFactory.CreateProtocol();
            clientDummy.Disconnected += client_disconnected;
            ClientDummies[clientDummy.Id] = clientDummy;

            fire_client_connected_event(clientDummy);
            e.Channel.Start();
        }

        private void client_disconnected(object sender, EventArgs e)
        {
            var client = (IClientDummy) sender;
            ClientDummies.Remove(client.Id);
            fire_client_disconnected_event(client);
        }

        private void fire_client_connected_event(IClientDummy theClient)
        {
            if (null != ClientConnected)
            {
                ClientConnected.Invoke(this, new ClientDummyEventArgs(theClient));
            }
        }

        private void fire_client_disconnected_event(IClientDummy theClient)
        {
            if (null != ClientDisconnected)
            {
                ClientDisconnected.Invoke(this, new ClientDummyEventArgs(theClient));
            }
        }
    }
}